#!/usr/bin/env bash
set -euo pipefail

# Install the Linux rendezvous/bootstrap server as a systemd service.
# Expects to be run on the destination host, with ./server present next to this script
# (e.g., extracted from punch-simple-linux-x86_64.tar.gz).

usage() {
  cat <<'EOF'
Usage:
  sudo ./install_rendezvous_service.sh [--install-dir DIR] [--port PORT]

Defaults:
  --install-dir /opt/punch-rendezvous
  --port        51820

What it does:
  - Creates user/group 'punch' (system account)
  - Copies server binary to install dir
  - Installs systemd unit 'punch-rendezvous.service'
  - Enables + starts the service

Logs:
  journalctl -u punch-rendezvous -f
EOF
}

INSTALL_DIR="/opt/punch-rendezvous"
PORT="51820"

while [[ $# -gt 0 ]]; do
  case "$1" in
    --install-dir) INSTALL_DIR="$2"; shift 2 ;;
    --port) PORT="$2"; shift 2 ;;
    -h|--help) usage; exit 0 ;;
    *) echo "Unknown arg: $1"; usage; exit 2 ;;
  esac
done

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

if [[ $EUID -ne 0 ]]; then
  echo "Must run as root (use sudo)."
  exit 1
fi

if [[ ! -f "$SCRIPT_DIR/../server" ]]; then
  echo "Expected server binary at: $SCRIPT_DIR/../server"
  echo "Run this from the extracted bundle: punch-simple/linux/install_rendezvous_service.sh"
  exit 1
fi

echo "[1/5] Creating system user/group (punch)..."
if ! id -u punch >/dev/null 2>&1; then
  useradd --system --home-dir "$INSTALL_DIR" --shell /usr/sbin/nologin punch
fi
if ! getent group punch >/dev/null 2>&1; then
  groupadd --system punch || true
fi

echo "[2/5] Installing files to $INSTALL_DIR..."
mkdir -p "$INSTALL_DIR"
cp -f "$SCRIPT_DIR/../server" "$INSTALL_DIR/server"
chmod 0755 "$INSTALL_DIR/server"
chown -R punch:punch "$INSTALL_DIR"

echo "[3/5] Installing systemd unit..."
UNIT_PATH="/etc/systemd/system/punch-rendezvous.service"
cat > "$UNIT_PATH" <<EOF
[Unit]
Description=Punch Rendezvous (QUIC bootstrap) server
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
User=punch
Group=punch
WorkingDirectory=$INSTALL_DIR
ExecStart=$INSTALL_DIR/server --listen-addr 0.0.0.0 --port $PORT
Restart=always
RestartSec=3
NoNewPrivileges=true
PrivateTmp=true
ProtectSystem=strict
ProtectHome=true
ReadWritePaths=$INSTALL_DIR

[Install]
WantedBy=multi-user.target
EOF

echo "[4/5] Enabling + starting service..."
systemctl daemon-reload
systemctl enable punch-rendezvous.service
systemctl restart punch-rendezvous.service

echo "[5/5] Status:"
systemctl --no-pager status punch-rendezvous.service || true
echo "Tail logs: journalctl -u punch-rendezvous -f"

